import os
import logging
import telebot
from telebot import types
from telebot import apihelper

# Enable middleware/logging as before
apihelper.ENABLE_MIDDLEWARE = True
logger = telebot.logger
telebot.logger.setLevel(logging.INFO)

# Prefer env var if provided, fallback to existing token
API_TOKEN = os.environ.get('BOT_TOKEN')
bot = telebot.TeleBot(API_TOKEN, parse_mode='HTML')

# ---- Database (SQLAlchemy) ----
from db import (
    save_resume as db_save_resume,
    get_active_resumes_by_county as db_get_active_resumes_by_county,
    get_unseen_resumes_by_county as db_get_unseen_resumes_by_county,
    mark_resumes_seen as db_mark_resumes_seen,
    save_job_post as db_save_job_post,
    get_active_jobs_by_county as db_jobs_by_county,
    get_unseen_jobs_by_county as db_unseen_jobs_by_county,
    mark_jobs_seen as db_mark_jobs_seen,
    increment_job_views as db_inc_job_views,
    get_remaining_views as db_get_remaining_views,
    add_views as db_add_views,
    cleanup_expired as db_cleanup_expired,
    get_my_resumes as db_get_my_resumes,
    get_my_jobs as db_get_my_jobs,
    delete_my_resume as db_delete_my_resume,
    delete_my_job as db_delete_my_job,
    # Admin helpers (username-based)
    is_admin_username as db_is_admin_username,
    add_admin_username as db_add_admin_username,
    remove_admin_username as db_remove_admin_username,
    list_admin_usernames as db_list_admin_usernames,
    # Settings / Night mode
    get_night_mode_config as db_get_night_cfg,
    set_setting as db_set_setting,
    # Promo posts
    save_promo_post as db_save_promo_post,
    get_random_promo_post as db_get_random_promo,
    list_promo_posts as db_list_promos,
    delete_promo_post as db_delete_promo,
    increment_promo_views as db_inc_promo_views,
)

# ---- Admin / Owner ----
OWNER_USERNAME = os.environ.get('OWNER_USERNAME').strip().lstrip('@').lower()


def is_owner_or_admin_username(username: str | None) -> bool:
    if not username:
        return False
    uname = username.lstrip('@').lower()
    return (OWNER_USERNAME and uname == OWNER_USERNAME) or db_is_admin_username(uname)


def night_blocking_active_for(username: str | None) -> bool:
    """True if night mode is enabled and we should block this (non-admin) user."""
    cfg = db_get_night_cfg()
    if not cfg.get('enabled'):
        return False
    # Admins bypass
    if is_owner_or_admin_username(username):
        return False
    try:
        from datetime import datetime, time as t
        now = datetime.now().time()
        s, e = cfg.get('start'), cfg.get('end')
        if not s or not e:
            return False
        sh, sm = [int(x) for x in s.split(':')]
        eh, em = [int(x) for x in e.split(':')]
        st = t(sh, sm)
        et = t(eh, em)
        return (st <= now <= et) if st <= et else (now >= st or now <= et)
    except Exception:
        return False


# ---- Keyboards ----
MAIN_KB = types.ReplyKeyboardMarkup(resize_keyboard=True, row_width=2)
MAIN_KB.add(types.KeyboardButton('کارفرما هستم'), types.KeyboardButton('متقاضی کار هستم'))
MAIN_KB.add(types.KeyboardButton('مدیریت'))

APPLICANT_KB = types.ReplyKeyboardMarkup(resize_keyboard=True, row_width=2)
APPLICANT_KB.add(
    types.KeyboardButton('ارسال رزومه'),
    types.KeyboardButton('مشاهده آگهی استخدام'),
)
APPLICANT_KB.add(
    types.KeyboardButton('رزومه‌های من'),
)
APPLICANT_KB.add(types.KeyboardButton('بازگشت'))

EMPLOYER_KB = types.ReplyKeyboardMarkup(resize_keyboard=True, row_width=2)
EMPLOYER_KB.add(
    types.KeyboardButton('ارسال آگهی استخدام'),
    types.KeyboardButton('مشاهده'),
)
EMPLOYER_KB.add(
    types.KeyboardButton('آگهی‌های من'),
)
EMPLOYER_KB.add(types.KeyboardButton('بازگشت'))

# ---- Start Menu & Help ----
START_KB = types.ReplyKeyboardMarkup(resize_keyboard=True, row_width=2)
START_KB.add(types.KeyboardButton('شروع'))

HELP_TEXT = """راهنمای استفاده از ربات:

روی دکمه شروع کلیک کنید...
۱) متقاضی کار هستید:
- از منوی «متقاضی کار هستم» رزومه خود را ثبت کنید.
- برای مشاهده آگهی‌های استخدام، شهرستان و دسته‌بندی را انتخاب کنید. روزانه تا ۳ آگهی نمایش داده می‌شود (اگر بیش از ۵ آگهی فعال وجود داشته باشد).

۲) کارفرما هستید:
- از منوی «کارفرما هستم» آگهی استخدام ثبت کنید.
- رزومه‌های فعال را بر اساس شهرستان و دسته‌بندی ببینید (نمایش تا ۳ رزومه در روز در صورت بیش از ۵ رزومه فعال).

نکته: آگهی‌ها و رزومه‌ها به‌صورت خودکار پس از یک هفته منقضی می‌شوند. برای بازگشت به منو از دکمه‌های کیبورد استفاده کنید."""

CATEGORY_KB = types.ReplyKeyboardMarkup(resize_keyboard=True, row_width=2)
CATEGORY_KB.add(
    types.KeyboardButton('کارگر ساده'),
    types.KeyboardButton('کارگر ماهر (صنعت کار)'),
    types.KeyboardButton('اداری'),
    types.KeyboardButton('متخصص فنی مهندسی'),
    types.KeyboardButton('بهداشت حرفه‌ای و ایمنی کار'),
)
CATEGORY_KB.add(types.KeyboardButton('بازگشت'))

# ---- Admin Keyboard ----
ADMIN_KB = types.ReplyKeyboardMarkup(resize_keyboard=True, row_width=2)
ADMIN_KB.add(
    types.KeyboardButton('افزودن ادمین'),
    types.KeyboardButton('حذف ادمین'),
)
ADMIN_KB.add(types.KeyboardButton('لیست ادمین‌ها'))
ADMIN_KB.add(
    types.KeyboardButton('تنظیم حالت شب'),
)
ADMIN_KB.add(types.KeyboardButton(' ایجاد پست تبلیغاتی '))
ADMIN_KB.add(types.KeyboardButton('لیست پست‌های تبلیغاتی'))
ADMIN_KB.add(types.KeyboardButton('بازگشت'))

# ---- Simple in-memory state ----
# NOTE: If you deploy horizontally, replace with persistent storage.
user_state = {}
# state shape: {
#   chat_id: {
#       'flow': 'resume' | 'jobs' | None,
#       'county': str | None,
#       'category': str | None,
#       'awaiting': 'county' | 'category' | 'resume_text' | None
#   }
# }

# Deprecated in-memory resumes store (migrated to DB)
# RESUMES moved to database; see db.py

# County list (sample for Khorasan Razavi). Extend as needed.
COUNTIES = [
    'مشهد', 'نیشابور', 'سبزوار', 'تربت حیدریه', 'قوچان', 'کاشمر',
    'گناباد', 'تایباد', 'طرقبه شاندیز', 'چناران', 'بردسکن', 'خواف',
    'سرخس', 'درگز', 'فیروزه', 'خلیل‌آباد', 'کلات', 'جوین', 'جغتای'
]
COUNTY_PAGE_SIZE = 6

# ---- Jobs are now provided by DB (EmployerJob). Sample list removed. ----
JOBS = []

# user daily views: {chat_id: {date: 'YYYY-MM-DD', count: int}}
user_daily_views = {}

# Employer jobs moved to database; see db.py


def today_key():
    from datetime import datetime
    return datetime.now().strftime('%Y-%m-%d')


def get_remaining_views(chat_id: int) -> int:
    v = user_daily_views.get(chat_id)
    t = today_key()
    if not v or v.get('date') != t:
        user_daily_views[chat_id] = {'date': t, 'count': 0}
        return 3
    return max(0, 3 - v['count'])


def add_views(chat_id: int, n: int) -> None:
    t = today_key()
    v = user_daily_views.get(chat_id)
    if not v or v.get('date') != t:
        user_daily_views[chat_id] = {'date': t, 'count': 0}
    user_daily_views[chat_id]['count'] = min(3, user_daily_views[chat_id]['count'] + n)


def get_active_jobs_by_county(county: str):
    return [j for j in JOBS if j.get('active') and j.get('county') == county]


def format_job(job: dict) -> str:
    # Jobs from DB have 'text' not 'desc' and no separate 'title'
    text = job.get('text', '-')
    county = job.get('county', '-')
    category = job.get('category', '-')
    created = job.get('created_at', '-')
    return f'شهرستان: {county}\nدسته‌بندی: {category}\nتاریخ ثبت: {created}\n\n{text}'


def format_resume(resume: dict) -> str:
    # For display to employer when viewing resumes
    text = resume.get('text', '-')
    county = resume.get('county', '-')
    category = resume.get('category', '-')
    created_at = resume.get('created_at', '-')
    return f'شهرستان: {county}\nدسته‌بندی: {category}\nتاریخ ثبت: {created_at}\n\n{text}'


def format_my_job(job: dict) -> str:
    # Include status and views for owner's perspective
    base = format_job(job)
    status = 'فعال' if job.get('active') else 'غیرفعال'
    views = job.get('views', 0)
    jid = job.get('id', '-')
    return f'شناسه آگهی: {jid}\nوضعیت: {status}\nبازدید: {views}\n\n{base}'


def send_jobs_for_county(chat_id: int, county: str):
    # housekeeping: purge expired records (> 7 days)
    try:
        db_cleanup_expired(7)
    except Exception:
        pass
    total = len(db_jobs_by_county(county))
    bot.send_message(chat_id, f'تعداد آگهی‌های فعال در "{county}": {total}')

    # Only show ads if more than 5 active ads exist
    if total > 5:
        remaining = db_get_remaining_views(chat_id)
        if remaining <= 0:
            bot.send_message(chat_id, 'سقف مشاهده روزانه شما (۳ آگهی) به اتمام رسیده است. لطفاً فردا مجدداً تلاش کنید.', reply_markup=APPLICANT_KB)
            return
        unseen = db_unseen_jobs_by_county(chat_id, county)
        if not unseen:
            bot.send_message(chat_id, 'آگهی جدیدی برای نمایش وجود ندارد.', reply_markup=APPLICANT_KB)
            return
        n = min(3, remaining)
        to_show = unseen[:n]
        for job in to_show:
            bot.send_message(chat_id, format_job(job))
        # update daily user views, ad views, and mark as seen
        ids = [j['id'] for j in to_show]
        db_add_views(chat_id, len(ids))
        db_inc_job_views(ids, inc=1)
        db_mark_jobs_seen(chat_id, ids)
        left = db_get_remaining_views(chat_id)
        bot.send_message(chat_id, f'تعداد آگهی‌های نمایش‌داده‌شده: {len(ids)}. باقیمانده امروز: {left} از ۳.', reply_markup=APPLICANT_KB)
    else:
        # Do not show ads, only count
        bot.send_message(chat_id, 'به‌دلیل کمتر بودن تعداد آگهی‌های فعال از ۶ مورد، نمایش آگهی انجام نشد.', reply_markup=APPLICANT_KB)


def get_user(chat_id: int):
    if chat_id not in user_state:
        user_state[chat_id] = {'flow': None, 'county': None, 'awaiting': None, 'category': None}
    # ensure keys exist
    st = user_state[chat_id]
    st.setdefault('flow', None)
    st.setdefault('county', None)
    st.setdefault('awaiting', None)
    st.setdefault('category', None)
    return st


def county_inline_keyboard(page: int = 0) -> types.ReplyKeyboardMarkup:
    # Switched to ReplyKeyboard for scrollable list (no pagination)
    kb = types.ReplyKeyboardMarkup(resize_keyboard=True, row_width=2)
    # Add county buttons (reply keyboard scrolls vertically)
    kb.add(*[types.KeyboardButton(name) for name in COUNTIES])
    # Add cancel button
    kb.add(types.KeyboardButton('انصراف'))
    return kb
def send_advertis(chat_id):
    try:
        promo = db_get_random_promo()
        if promo:
            pid = promo['id']
            
            if promo['media_type'] == 'photo' and promo.get('file_id'):
                cap = (promo.get('caption') or '').strip() or '—'
                cap = f"{cap}\n"
                bot.send_photo(chat_id, promo['file_id'], caption=cap)
                db_inc_promo_views(pid, 1)
            elif promo['media_type'] == 'video' and promo.get('file_id'):
                cap = (promo.get('caption') or '').strip() or '—'
                cap = f"{cap}"
                bot.send_video(chat_id, promo['file_id'], caption=cap)
                db_inc_promo_views(pid, 1)
            elif promo['media_type'] == 'text' and promo.get('text'):
                text = f"{promo['text']}\n"
                bot.send_message(chat_id, text)
                db_inc_promo_views(pid, 1)
    except Exception as e:
        logging.warning(f"Failed to send promo on start: {e}")

# ---- Handlers ----
@bot.message_handler(commands=['start'])
def handle_start(message: types.Message):
    chat_id = message.chat.id
    # Night-mode gate at start
    if night_blocking_active_for(getattr(message.from_user, 'username', None)):
        bot.send_message(chat_id, 'ربات در حالت شب است و موقتاً امکان استفاده وجود ندارد. لطفاً بعداً تلاش کنید.')
        return
    # housekeeping: purge expired records (> 7 days)
    try:
        stats = db_cleanup_expired(7)
        logging.info(f"Cleanup on /start: {stats}")
    except Exception as e:
        logging.warning(f"Cleanup failed: {e}")
    user_state[chat_id] = {'flow': None, 'county': None, 'awaiting': None, 'category': None, 'last_menu': 'start'}
    bot.send_message(chat_id, 'سلام! به ربات --استخدام صنعت خراسان-- خوش آمدید.\n\nبرای شروع «شروع» را انتخاب کنید یا برای آشنایی با عملکرد ربات، «راهنما» را از menu انتخاب کنید.', reply_markup=START_KB)
    # bot.send_message(chat_id, HELP_TEXT, reply_markup=START_KB)

    # After intro, send a random promo post if available
    send_advertis(chat_id)
@bot.message_handler(commands=['help'])
def handle_help_cmd(message: types.Message):
    chat_id = message.chat.id
    bot.send_message(chat_id, HELP_TEXT, reply_markup=START_KB)



@bot.message_handler(func=lambda m: m.text == 'شروع')
def handle_go_main_menu(message: types.Message):
    chat_id = message.chat.id
    st = get_user(chat_id)
    st['last_menu'] = 'main'
    bot.send_message(chat_id, 'لطفاً یک گزینه را انتخاب کنید:', reply_markup=MAIN_KB)


@bot.message_handler(func=lambda m: m.text == 'بازگشت')
def handle_back(message: types.Message):
    chat_id = message.chat.id
    user_state[chat_id] = {'flow': None, 'county': None, 'awaiting': None, 'category': None}
    bot.send_message(chat_id, 'به منوی اصلی بازگشتید.', reply_markup=MAIN_KB)


@bot.message_handler(func=lambda m: m.text == 'متقاضی کار هستم')
def handle_applicant(message: types.Message):
    chat_id = message.chat.id
    st = get_user(chat_id)
    st['flow'] = None
    st['county'] = None
    st['awaiting'] = None
    st['category'] = None
    st['last_menu'] = 'applicant'
    bot.send_message(chat_id, 'لطفاً یک گزینه را انتخاب کنید:', reply_markup=APPLICANT_KB)


@bot.message_handler(func=lambda m: m.text == 'کارفرما هستم')
def handle_employer(message: types.Message):
    chat_id = message.chat.id
    st = get_user(chat_id)
    st['flow'] = None
    st['county'] = None
    st['awaiting'] = None
    st['category'] = None
    st['last_menu'] = 'employer'
    bot.send_message(chat_id, 'لطفاً یک گزینه را انتخاب کنید:', reply_markup=EMPLOYER_KB)


@bot.message_handler(func=lambda m: m.text == 'مدیریت')
def handle_admin_entry(message: types.Message):
    chat_id = message.chat.id
    if not is_owner_or_admin_username(getattr(message.from_user, 'username', None)):
        bot.send_message(chat_id, 'دسترسی شما به این بخش مجاز نیست.', reply_markup=MAIN_KB)
        return
    st = get_user(chat_id)
    st['flow'] = 'admin'
    st['awaiting'] = None
    st['last_menu'] = 'admin'

    # Show night mode current status
    cfg = db_get_night_cfg()
    status = 'فعال' if cfg.get('enabled') else 'غیرفعال'
    start = cfg.get('start') or '--'
    end = cfg.get('end') or '--'
    info = f"حالت شب: {status}\nشروع: {start}\nپایان: {end}"

    bot.send_message(chat_id, 'پنل مدیریت: یکی از گزینه‌ها را انتخاب کنید.\n\n' + info, reply_markup=ADMIN_KB)


@bot.message_handler(func=lambda m: m.text == 'افزودن ادمین')
def handle_admin_add(message: types.Message):
    chat_id = message.chat.id
    if not is_owner_or_admin_username(getattr(message.from_user, 'username', None)):
        bot.send_message(chat_id, 'دسترسی شما به این بخش مجاز نیست.', reply_markup=MAIN_KB)
        return
    # Only owner can add new admins
    if getattr(message.from_user, 'username', None) is None or message.from_user.username.lstrip('@').lower() != OWNER_USERNAME:
        bot.send_message(chat_id, 'فقط مالک ربات می‌تواند ادمین جدید اضافه کند.', reply_markup=ADMIN_KB)
        return
    st = get_user(chat_id)
    st['flow'] = 'admin'
    st['awaiting'] = 'add_admin_username'
    bot.send_message(chat_id, 'نام کاربری فرد (بدون @) را ارسال کنید:', reply_markup=types.ReplyKeyboardRemove())


@bot.message_handler(func=lambda m: m.text == 'تنظیم حالت شب')
def handle_night_mode_menu(message: types.Message):
    chat_id = message.chat.id
    # Owner and admins can manage
    if not is_owner_or_admin_username(getattr(message.from_user, 'username', None)):
        bot.send_message(chat_id, 'دسترسی شما به این بخش مجاز نیست.', reply_markup=MAIN_KB)
        return
    kb = types.ReplyKeyboardMarkup(resize_keyboard=True, row_width=2)
    kb.add(types.KeyboardButton('فعال/غیرفعال حالت شب'))
    kb.add(types.KeyboardButton('تنظیم ساعت شروع (HH:MM)'), types.KeyboardButton('تنظیم ساعت پایان (HH:MM)'))
    kb.add(types.KeyboardButton('بازگشت'))
    cfg = db_get_night_cfg()
    status = 'فعال' if cfg.get('enabled') else 'غیرفعال'
    start = cfg.get('start') or '--'
    end = cfg.get('end') or '--'
    text = f"حالت شب: {status}\nشروع: {start}\nپایان: {end}\n\nیک گزینه را انتخاب کنید:"
    bot.send_message(chat_id, text, reply_markup=kb)


@bot.message_handler(func=lambda m: m.text == 'ایجاد پست تبلیغاتی')
def handle_promo_menu(message: types.Message):
    chat_id = message.chat.id
    if not is_owner_or_admin_username(getattr(message.from_user, 'username', None)):
        bot.send_message(chat_id, 'دسترسی شما به این بخش مجاز نیست.', reply_markup=MAIN_KB)
        return
    st = get_user(chat_id)
    st['flow'] = 'admin_promo'
    st['awaiting'] = None
    kb = types.ReplyKeyboardMarkup(resize_keyboard=True, row_width=2)
    kb.add(types.KeyboardButton('تبلیغ: عکس + کپشن'))
    kb.add(types.KeyboardButton('تبلیغ: ویدیو + کپشن'))
    kb.add(types.KeyboardButton('تبلیغ: متن تنها'))
    kb.add(types.KeyboardButton('بازگشت'))
    bot.send_message(chat_id, 'نوع پست تبلیغاتی را انتخاب کنید:', reply_markup=kb)


@bot.message_handler(func=lambda m: m.text == 'لیست پست‌های تبلیغاتی')
def handle_list_promos(message: types.Message):
    chat_id = message.chat.id
    if not is_owner_or_admin_username(getattr(message.from_user, 'username', None)):
        bot.send_message(chat_id, 'دسترسی شما به این بخش مجاز نیست.', reply_markup=MAIN_KB)
        return
    items = db_list_promos(limit=50)
    if not items:
        bot.send_message(chat_id, 'هیچ پست تبلیغاتی ثبت نشده است.', reply_markup=ADMIN_KB)
        return
    bot.send_message(chat_id, f'تعداد پست‌های تبلیغاتی: {len(items)}')
    for it in items[:10]:  # limit messages to avoid spam
        kb = types.InlineKeyboardMarkup()
        kb.add(types.InlineKeyboardButton('حذف', callback_data=f'del_promo:{it["id"]}'))
        mt = it.get('media_type')
        file_id = it.get('file_id')
        if mt == 'photo' and file_id:
            cap = (it.get('caption') or '').strip()
            cap = cap if cap else '—'
            cap = f"{cap}\n— شناسه: {it['id']} | تاریخ: {it['created_at']} | بازدید: {it.get('views', 0)}"
            bot.send_photo(chat_id, file_id, caption=cap, reply_markup=kb)
        elif mt == 'video' and file_id:
            cap = (it.get('caption') or '').strip()
            cap = cap if cap else '—'
            cap = f"{cap}\n— شناسه: {it['id']} | تاریخ: {it['created_at']} | بازدید: {it.get('views', 0)}"
            bot.send_video(chat_id, file_id, caption=cap, reply_markup=kb)
        else:
            # Text-only or missing media: show preview message
            preview = it.get('text') or it.get('caption') or ''
            preview = (preview[:200] + '…') if preview and len(preview) > 200 else (preview or '—')
            text = f"شناسه: {it['id']} | نوع: {mt} | تاریخ: {it['created_at']} | بازدید: {it.get('views', 0)}\n{preview}"
            bot.send_message(chat_id, text, reply_markup=kb)
    if len(items) > 10:
        bot.send_message(chat_id, '— تنها ۱۰ مورد اخیر نمایش داده شد.', reply_markup=ADMIN_KB)


@bot.message_handler(func=lambda m: m.text in {'تبلیغ: عکس + کپشن', 'تبلیغ: ویدیو + کپشن', 'تبلیغ: متن تنها'})
def handle_promo_type(message: types.Message):
    chat_id = message.chat.id
    if not is_owner_or_admin_username(getattr(message.from_user, 'username', None)):
        bot.send_message(chat_id, 'دسترسی شما به این بخش مجاز نیست.', reply_markup=MAIN_KB)
        return
    st = get_user(chat_id)
    st['flow'] = 'admin_promo'
    t = message.text
    if t == 'تبلیغ: عکس + کپشن':
        st['awaiting'] = 'promo_photo'
        bot.send_message(chat_id, 'لطفاً یک عکس به همراه کپشن ارسال کنید.', reply_markup=types.ReplyKeyboardRemove())
    elif t == 'تبلیغ: ویدیو + کپشن':
        st['awaiting'] = 'promo_video'
        bot.send_message(chat_id, 'لطفاً یک ویدیو به همراه کپشن ارسال کنید.', reply_markup=types.ReplyKeyboardRemove())
    else:
        st['awaiting'] = 'promo_text'
        bot.send_message(chat_id, 'متن پست تبلیغاتی را ارسال کنید.', reply_markup=types.ReplyKeyboardRemove())


@bot.message_handler(func=lambda m: m.text == 'فعال/غیرفعال حالت شب')
def handle_toggle_night(message: types.Message):
    chat_id = message.chat.id
    if not is_owner_or_admin_username(getattr(message.from_user, 'username', None)):
        bot.send_message(chat_id, 'دسترسی شما به این بخش مجاز نیست.', reply_markup=MAIN_KB)
        return
    st = get_user(chat_id)
    st['flow'] = 'admin'
    st['awaiting'] = 'night_mode_toggle'
    bot.send_message(chat_id, 'حالت شب را مشخص کنید: «فعال» یا «غیرفعال»', reply_markup=types.ReplyKeyboardRemove())


@bot.message_handler(func=lambda m: m.text == 'تنظیم ساعت شروع (HH:MM)')
def handle_night_start(message: types.Message):
    chat_id = message.chat.id
    if not is_owner_or_admin_username(getattr(message.from_user, 'username', None)):
        bot.send_message(chat_id, 'دسترسی شما به این بخش مجاز نیست.', reply_markup=MAIN_KB)
        return
    st = get_user(chat_id)
    st['flow'] = 'admin'
    st['awaiting'] = 'night_mode_start'
    bot.send_message(chat_id, 'لطفاً ساعت شروع را به‌صورت 24 ساعته و در قالب HH:MM وارد کنید. مثال: 22:30', reply_markup=types.ReplyKeyboardRemove())


@bot.message_handler(func=lambda m: m.text == 'تنظیم ساعت پایان (HH:MM)')
def handle_night_end(message: types.Message):
    chat_id = message.chat.id
    if not is_owner_or_admin_username(getattr(message.from_user, 'username', None)):
        bot.send_message(chat_id, 'دسترسی شما به این بخش مجاز نیست.', reply_markup=MAIN_KB)
        return
    st = get_user(chat_id)
    st['flow'] = 'admin'
    st['awaiting'] = 'night_mode_end'
    bot.send_message(chat_id, 'لطفاً ساعت پایان را به‌صورت 24 ساعته و در قالب HH:MM وارد کنید. مثال: 06:30', reply_markup=types.ReplyKeyboardRemove())


@bot.message_handler(func=lambda m: m.text == 'حذف ادمین')
def handle_admin_remove(message: types.Message):
    chat_id = message.chat.id
    if not is_owner_or_admin_username(getattr(message.from_user, 'username', None)):
        bot.send_message(chat_id, 'دسترسی شما به این بخش مجاز نیست.', reply_markup=MAIN_KB)
        return
    # Only owner can remove admins
    if getattr(message.from_user, 'username', None) is None or message.from_user.username.lstrip('@').lower() != OWNER_USERNAME:
        bot.send_message(chat_id, 'فقط مالک ربات می‌تواند ادمین را حذف کند.', reply_markup=ADMIN_KB)
        return
    st = get_user(chat_id)
    st['flow'] = 'admin'
    st['awaiting'] = 'remove_admin_username'
    bot.send_message(chat_id, 'نام کاربری ادمین برای حذف (بدون @) را ارسال کنید:', reply_markup=types.ReplyKeyboardRemove())


@bot.message_handler(func=lambda m: m.text == 'لیست ادمین‌ها')
def handle_admin_list(message: types.Message):
    chat_id = message.chat.id
    if not is_owner_or_admin_username(getattr(message.from_user, 'username', None)):
        bot.send_message(chat_id, 'دسترسی شما به این بخش مجاز نیست.', reply_markup=MAIN_KB)
        return
    admins = db_list_admin_usernames()
    admins_display = []
    if OWNER_USERNAME:
        admins_display.append(OWNER_USERNAME + ' (مالک)')
    admins_display += [('@' + a) for a in admins if a != OWNER_USERNAME]
    cfg = db_get_night_cfg()
    status = 'فعال' if cfg.get('enabled') else 'غیرفعال'
    start = cfg.get('start') or '--'
    end = cfg.get('end') or '--'
    nm = f"\n\nحالت شب: {status}\nشروع: {start}\nپایان: {end}"
    text = 'ادمین‌های ثبت‌شده:\n' + ('\n'.join(admins_display) if admins_display else 'هیچ ادمینی ثبت نشده است.') + nm
    bot.send_message(chat_id, text, reply_markup=ADMIN_KB)


@bot.message_handler(func=lambda m: m.text == 'ارسال رزومه')
def handle_resume(message: types.Message):
    chat_id = message.chat.id
    st = get_user(chat_id)
    st['flow'] = 'resume'
    st['county'] = None
    st['awaiting'] = 'county'

    # 1) Remove reply keyboard (main/applicant menu)
    bot.send_message(
        chat_id,
        'ابتدا شهرستان خود را انتخاب کنید:',
        reply_markup=types.ReplyKeyboardRemove()
    )
    # 2) Show county list as reply keyboard (scrollable)
    bot.send_message(
        chat_id,
        'از کیبورد زیر شهرستان را انتخاب کنید:',
        reply_markup=county_inline_keyboard(page=0)
    )


@bot.callback_query_handler(func=lambda c: c.data and (c.data.startswith('county_page:') or c.data.startswith('county:') or c.data == 'county_cancel' or c.data.startswith('del_resume:') or c.data.startswith('del_job:') or c.data.startswith('del_promo:')))
def handle_county_callbacks(call: types.CallbackQuery):
    chat_id = call.message.chat.id
    st = get_user(chat_id)

    data = call.data or ''

    # Deletion callbacks
    if data.startswith('del_resume:'):
        try:
            rid = int(data.split(':', 1)[1])
        except Exception:
            bot.answer_callback_query(call.id, 'شناسه نامعتبر است.')
            return
        ok = db_delete_my_resume(chat_id, rid)
        msg = 'رزومه حذف شد.' if ok else 'حذف انجام نشد (ممکن است این رزومه متعلق به شما نباشد یا وجود نداشته باشد).'
        try:
            bot.answer_callback_query(call.id, msg)
        except Exception:
            pass
        bot.send_message(chat_id, msg, reply_markup=APPLICANT_KB)
        return

    if data.startswith('del_job:'):
        try:
            jid = int(data.split(':', 1)[1])
        except Exception:
            bot.answer_callback_query(call.id, 'شناسه نامعتبر است.')
            return
        ok = db_delete_my_job(chat_id, jid)
        msg = 'آگهی حذف شد.' if ok else 'حذف انجام نشد (ممکن است این آگهی متعلق به شما نباشد یا وجود نداشته باشد).'
        try:
            bot.answer_callback_query(call.id, msg)
        except Exception:
            pass
        bot.send_message(chat_id, msg, reply_markup=EMPLOYER_KB)
        return

    if data.startswith('del_promo:'):
        # Only owner/admin can delete promos
        if not is_owner_or_admin_username(getattr(call.from_user, 'username', None)):
            try:
                bot.answer_callback_query(call.id, 'مجوز کافی ندارید.')
            except Exception:
                pass
            return
        try:
            pid = int(data.split(':', 1)[1])
        except Exception:
            bot.answer_callback_query(call.id, 'شناسه نامعتبر است.')
            return
        ok = db_delete_promo(pid)
        msg = 'پست تبلیغاتی حذف شد.' if ok else 'حذف انجام نشد (ممکن است وجود نداشته باشد).'
        try:
            bot.answer_callback_query(call.id, msg)
        except Exception:
            pass
        bot.send_message(chat_id, msg, reply_markup=ADMIN_KB)
        return

    # Legacy inline keyboard callbacks are no longer used. Guide user to new reply keyboard.
    try:
        bot.answer_callback_query(call.id, 'لیست شهرستان‌ها اکنون به‌صورت قابل اسکرول نمایش داده می‌شود.')
    except Exception:
        pass

    # If user cancels from legacy UI, honor it
    if data == 'county_cancel':
        st['awaiting'] = None
        st['flow'] = None
        kb = EMPLOYER_KB if st.get('last_menu') == 'employer' else APPLICANT_KB
        bot.send_message(chat_id, 'عملیات انتخاب شهرستان لغو شد.', reply_markup=kb)
        return

    # Prompt user to use the new reply keyboard
    bot.send_message(chat_id, 'از کیبورد زیر شهرستان خود را انتخاب کنید:', reply_markup=county_inline_keyboard(page=0))
    return


# County selection via ReplyKeyboard
@bot.message_handler(func=lambda m: (get_user(m.chat.id).get('awaiting') == 'county') and (m.text in COUNTIES or m.text == 'انصراف'))
def handle_county_reply(message: types.Message):
    chat_id = message.chat.id
    st = get_user(chat_id)
    # Handle cancel
    if message.text == 'انصراف':
        st['awaiting'] = None
        st['flow'] = None
        kb = EMPLOYER_KB if st.get('last_menu') == 'employer' else APPLICANT_KB
        bot.send_message(chat_id, 'عملیات انتخاب شهرستان لغو شد.', reply_markup=kb)
        return

    county_name = message.text
    st['county'] = county_name
    if st.get('flow') == 'resume':
        st['awaiting'] = 'category'
        bot.send_message(chat_id, f'شهرستان "{county_name}" ثبت شد. حالا دسته‌بندی شغلی را انتخاب کنید:', reply_markup=CATEGORY_KB)
    elif st.get('flow') == 'jobs' and st.get('awaiting') == 'county':
        st['awaiting'] = None
        send_jobs_for_county(chat_id, county_name)
    elif st.get('flow') == 'post_job' and st.get('awaiting') == 'county':
        st['awaiting'] = 'category'
        bot.send_message(chat_id, f'شهرستان "{county_name}" ثبت شد. حالا دسته‌بندی شغلی را انتخاب کنید:', reply_markup=CATEGORY_KB)
    elif st.get('flow') == 'view_resumes' and st.get('awaiting') == 'county':
        st['awaiting'] = None
        send_resumes_for_county(chat_id, county_name)
    else:
        st['awaiting'] = None
        bot.send_message(chat_id, 'گزینه نامعتبر. لطفاً دوباره تلاش کنید.', reply_markup=APPLICANT_KB)


@bot.message_handler(func=lambda m: m.text == 'مشاهده آگهی استخدام')
def handle_view_jobs(message: types.Message):
    chat_id = message.chat.id
    st = get_user(chat_id)
    st['flow'] = 'jobs'
    st['county'] = None
    st['awaiting'] = 'county'
    st['last_menu'] = 'applicant'

    # Ask for county first (reply keyboard)
    bot.send_message(
        chat_id,
        'برای مشاهده آگهی‌ها، ابتدا شهرستان خود را انتخاب کنید:',
        reply_markup=types.ReplyKeyboardRemove()
    )
    bot.send_message(
        chat_id,
        'از کیبورد زیر شهرستان را انتخاب کنید:',
        reply_markup=county_inline_keyboard(page=0)
    )


@bot.message_handler(func=lambda m: m.text == 'رزومه‌های من')
def handle_my_resumes(message: types.Message):
    chat_id = message.chat.id
    items = db_get_my_resumes(chat_id)
    if not items:
        bot.send_message(chat_id, 'هیچ رزومه‌ای برای شما ثبت نشده است.', reply_markup=APPLICANT_KB)
        return
    bot.send_message(chat_id, f'تعداد رزومه‌های شما: {len(items)}')
    # Show up to 5 latest to avoid flooding
    for r in items[:5]:
        kb = types.InlineKeyboardMarkup()
        kb.add(types.InlineKeyboardButton('حذف', callback_data=f'del_resume:{r["id"]}'))
        bot.send_message(chat_id, format_resume(r), reply_markup=kb)
    if len(items) > 5:
        bot.send_message(chat_id, '— تنها ۵ مورد اخیر نمایش داده شد.', reply_markup=APPLICANT_KB)
    else:
        bot.send_message(chat_id, 'پایان لیست رزومه‌ها.', reply_markup=APPLICANT_KB)


@bot.message_handler(func=lambda m: m.text == 'ارسال آگهی استخدام')
def handle_post_job(message: types.Message):
    chat_id = message.chat.id
    st = get_user(chat_id)
    st['flow'] = 'post_job'
    st['county'] = None
    st['awaiting'] = 'county'
    st['last_menu'] = 'employer'

    bot.send_message(
        chat_id,
        'برای ارسال آگهی، ابتدا شهرستان خود را انتخاب کنید:',
        reply_markup=types.ReplyKeyboardRemove()
    )
    bot.send_message(
        chat_id,
        'از کیبورد زیر شهرستان را انتخاب کنید:',
        reply_markup=county_inline_keyboard(page=0)
    )


@bot.message_handler(func=lambda m: m.text == 'مشاهده')
def handle_employer_view(message: types.Message):
    chat_id = message.chat.id
    st = get_user(chat_id)
    st['flow'] = 'view_resumes'
    st['county'] = None
    st['awaiting'] = 'county'
    st['last_menu'] = 'employer'

    bot.send_message(
        chat_id,
        'برای مشاهده رزومه‌ها، ابتدا شهرستان خود را انتخاب کنید:',
        reply_markup=types.ReplyKeyboardRemove()
    )
    bot.send_message(
        chat_id,
        'از فهرست زیر شهرستان را انتخاب کنید:',
        reply_markup=county_inline_keyboard(page=0)
    )


@bot.message_handler(func=lambda m: m.text == 'آگهی‌های من')
def handle_my_jobs(message: types.Message):
    chat_id = message.chat.id
    items = db_get_my_jobs(chat_id)
    if not items:
        bot.send_message(chat_id, 'هیچ آگهی‌ای برای شما ثبت نشده است.', reply_markup=EMPLOYER_KB)
        return
    bot.send_message(chat_id, f'تعداد آگهی‌های شما: {len(items)}')
    for j in items[:5]:
        kb = types.InlineKeyboardMarkup()
        kb.add(types.InlineKeyboardButton('حذف', callback_data=f'del_job:{j["id"]}'))
        bot.send_message(chat_id, format_my_job(j), reply_markup=kb)
    if len(items) > 5:
        bot.send_message(chat_id, '— تنها ۵ مورد اخیر نمایش داده شد.', reply_markup=EMPLOYER_KB)
    else:
        bot.send_message(chat_id, 'پایان لیست آگهی‌ها.', reply_markup=EMPLOYER_KB)


@bot.message_handler(func=lambda m: m.text in {'کارگر ساده', 'کارگر ماهر (صنعت کار)', 'اداری', 'متخصص فنی مهندسی', 'بهداشت حرفه‌ای و ایمنی کار'})
def handle_category(message: types.Message):
    chat_id = message.chat.id
    st = get_user(chat_id)

    if st.get('awaiting') != 'category' or not st.get('county'):
        return

    category = message.text
    county = st['county']
    st['category'] = category

    if st.get('flow') == 'resume':
        # Applicant flow: ask for resume text
        st['awaiting'] = 'resume_text'
        template = (
           
            'نام و نام‌خانوادگی: \n'
            'شماره تماس: \n'
            'سن: \n'
            f'شهرستان: {county}\n'
            f'دسته‌بندی شغلی: {category}\n'
            'سابقه کار مرتبط (سال/توضیح مختصر): \n'
            'مهارت‌ها و گواهی‌نامه‌ها: \n'
            'مدرک تحصیلی/رشته: \n'
            '(میلیون در ماه)حداقل حقوق/دستمزد مورد انتظار: \n'
            
        )
        bot.send_message(chat_id, 'دسته‌بندی انتخاب شد: ' + category)
        bot.send_message(chat_id,  'لطفاً رزومه خود را مطابق قالب زیر به‌صورت یک پیام متنی ارسال کنید \nبرای سهولت متن قالب را کپی کنید و مقادیر را وارد کنید:\n\n')
        
        bot.send_message(chat_id, template)
        bot.send_message(chat_id,'\n— توجه: اعتبار هر رزومه یک هفته است.')




    elif st.get('flow') == 'post_job':
        # Employer posting flow: ask for job ad text template
        st['awaiting'] = 'job_text'
        template = (
           
            'عنوان شغلی: \n'
            'نام شرکت/کارگاه: \n'
            f'شهرستان: {county}\n'
            f'دسته‌بندی شغلی: {category}\n'
            'شرح وظایف: \n'
            'شرایط احراز (مهارت/سابقه): \n'
            'نوع همکاری (تمام‌وقت/پاره‌وقت/شیفت): \n'
            '(میلیون در ماه)حقوق و مزایا: \n'
            'نحوه تماس (تلفن/واتساپ/لینک): \n'
            
        )
        bot.send_message(chat_id, 'دسته‌بندی انتخاب شد: ' + category)
        bot.send_message(chat_id,  'لطفاً آگهی استخدام را مطابق قالب زیر به‌صورت یک پیام متنی ارسال کنید \nبرای سهولت متن قالب را کپی کنید و مقادیر را وارد کنید:\n\n')
        
        bot.send_message(chat_id, template)
        bot.send_message(chat_id,'\n— توجه: اعتبار هر آگهی یک هفته است.')

@bot.message_handler(content_types=['photo'])
def handle_promo_photo_message(message: types.Message):
    chat_id = message.chat.id
    st = get_user(chat_id)
    if st.get('flow') == 'admin_promo' and st.get('awaiting') == 'promo_photo' and is_owner_or_admin_username(getattr(message.from_user, 'username', None)):
        if not getattr(message, 'photo', None):
            bot.reply_to(message, 'عکس دریافت نشد. دوباره تلاش کنید.')
            return
        file_id = message.photo[-1].file_id  # largest size
        caption = getattr(message, 'caption', None) or ''
        db_save_promo_post(chat_id, 'photo', file_id, caption, None)
        st['awaiting'] = None
        st['flow'] = 'admin'
        bot.send_message(chat_id, 'پست تبلیغاتی (عکس) ذخیره شد.', reply_markup=ADMIN_KB)

# @bot.message_handler(content_types=['video'])
# def handle_promo_video_message(message: types.Message):
#     chat_id = message.chat.id
#     st = get_user(chat_id)
#     if st.get('flow') == 'admin_promo' and st.get('awaiting') == 'promo_video' and is_owner_or_admin_username(getattr(message.from_user, 'username', None)):
#         if not getattr(message, 'video', None):
#             bot.reply_to(message, 'ویدیو دریافت نشد. دوباره تلاش کنید.')
#             return
#         file_id = message.video.file_id
#         caption = getattr(message, 'caption', None) or ''
#         db_save_promo_post(chat_id, 'video', file_id, caption, None)
#         st['awaiting'] = None
#         st['flow'] = 'admin'
#         bot.send_message(chat_id, 'پست تبلیغاتی (ویدیو) ذخیره شد.', reply_markup=ADMIN_KB)

#     elif st.get('flow') == 'post_job':
#         # Employer posting flow: ask for job ad text template
#         st['awaiting'] = 'job_text'
#         county = st.get('county') or ''
#         category = st.get('category') or ''
#         template = (
           
#             'عنوان شغلی: \n'
#             'نام شرکت/کارگاه: \n'
#             f'شهرستان: {county}\n'
#             f'دسته‌بندی شغلی: {category}\n'
#             'شرح وظایف: \n'
#             'شرایط احراز (مهارت/سابقه): \n'
#             'نوع همکاری (تمام‌وقت/پاره‌وقت/شیفت): \n'
#             '(میلیون در ماه)حقوق و مزایا: \n'
#             'نحوه تماس (تلفن/واتساپ/لینک): \n'
            
#         )
#         bot.send_message(chat_id, f'دسته‌بندی انتخاب شد: {category}')
#         bot.send_message(chat_id,  'لطفاً آگهی استخدام را مطابق قالب زیر به‌صورت یک پیام متنی ارسال کنید \nبرای سهولت متن قالب را کپی کنید و مقادیر را وارد کنید:\n\n')
        
#         bot.send_message(chat_id, template)
#         bot.send_message(chat_id,'\n— توجه: اعتبار هر آگهی یک هفته است.')


# Handler to capture resume text and save it
@bot.message_handler(content_types=['text'])
def handle_resume_text(message: types.Message):
    chat_id = message.chat.id
    st = get_user(chat_id)

    # Global night-mode gate: deny all non-admins during night window
    cfg = db_get_night_cfg()
    if cfg.get('enabled'):
        try:
            from datetime import datetime
            now = datetime.now().time()
            def _parse(hm: str):
                h, m = hm.split(':')
                from datetime import time
                return time(int(h), int(m))
            start_s, end_s = cfg.get('start'), cfg.get('end')
            if start_s and end_s:
                start_t = _parse(start_s)
                end_t = _parse(end_s)
                in_window = (start_t <= now <= end_t) if start_t <= end_t else (now >= start_t or now <= end_t)
                # Allow owner/admin to bypass to manage settings
                if in_window and not is_owner_or_admin_username(getattr(message.from_user, 'username', None)):
                    bot.reply_to(message, 'ربات در حالت شب است و موقتاً امکان ارسال پیام وجود ندارد. لطفاً بعداً تلاش کنید.')
                    return
        except Exception:
            pass  # If parsing fails, ignore gate to avoid lockout

    # Admin promo: text-only post
    if st.get('flow') == 'admin_promo' and st.get('awaiting') == 'promo_text' and is_owner_or_admin_username(getattr(message.from_user, 'username', None)):
        raw = (message.text or '').strip()
        if not raw or raw.startswith('/'):
            bot.reply_to(message, 'متن نامعتبر است. لطفاً متن معمولی ارسال کنید.')
            return
        db_save_promo_post(chat_id, 'text', None, None, raw)
        st['awaiting'] = None
        st['flow'] = 'admin'
        bot.send_message(chat_id, 'پست تبلیغاتی (متن) ذخیره شد.', reply_markup=ADMIN_KB)
        return

    # Admin flows: add/remove admin by username and night mode settings
    if st.get('flow') == 'admin' and st.get('awaiting') in {'add_admin_username', 'remove_admin_username', 'night_mode_toggle', 'night_mode_start', 'night_mode_end'}:
        raw = (message.text or '').strip()
        if st['awaiting'] == 'add_admin_username':
            uname = raw.lstrip('@').lower()
            if not uname or ' ' in uname:
                bot.reply_to(message, 'نام کاربری معتبر نیست. لطفاً فقط username بدون فاصله ارسال کنید.')
                return
            if OWNER_USERNAME and uname == OWNER_USERNAME:
                bot.send_message(chat_id, 'این کاربر مالک ربات است و نیازی به افزودن ندارد.', reply_markup=ADMIN_KB)
            else:
                ok = db_add_admin_username(uname)
                msg = 'ادمین با موفقیت اضافه شد.' if ok else 'این نام کاربری قبلاً به عنوان ادمین ثبت شده است.'
                bot.send_message(chat_id, msg, reply_markup=ADMIN_KB)
            st['awaiting'] = None
            return
        if st['awaiting'] == 'remove_admin_username':
            uname = raw.lstrip('@').lower()
            if OWNER_USERNAME and uname == OWNER_USERNAME:
                bot.send_message(chat_id, 'حذف مالک ربات مجاز نیست.', reply_markup=ADMIN_KB)
            else:
                ok = db_remove_admin_username(uname)
                msg = 'ادمین حذف شد.' if ok else 'چنین ادمینی یافت نشد.'
                bot.send_message(chat_id, msg, reply_markup=ADMIN_KB)
            st['awaiting'] = None
            return
        if st['awaiting'] == 'night_mode_toggle':
            norm = raw.replace('\u200c', '').strip().lower()
            if norm in {'فعال', 'on', '1', 'true', 'روشن'}:
                db_set_setting('night_enabled', '1')
                bot.send_message(chat_id, 'حالت شب فعال شد.', reply_markup=ADMIN_KB)
            elif norm in {'غیرفعال', 'off', '0', 'false', 'خاموش'}:
                db_set_setting('night_enabled', '0')
                bot.send_message(chat_id, 'حالت شب غیرفعال شد.', reply_markup=ADMIN_KB)
            else:
                bot.send_message(chat_id, 'پاسخ نامعتبر. یکی از گزینه‌ها: فعال / غیرفعال', reply_markup=ADMIN_KB)
            st['awaiting'] = None
            return
        if st['awaiting'] in {'night_mode_start', 'night_mode_end'}:
            # Expect HH:MM 24h
            import re
            if not re.match(r'^\d{2}:\d{2}$', raw):
                bot.send_message(chat_id, 'فرمت زمان نامعتبر است. مثال صحیح: 22:30', reply_markup=ADMIN_KB)
                st['awaiting'] = None
                return
            key = 'night_start' if st['awaiting'] == 'night_mode_start' else 'night_end'
            db_set_setting(key, raw)
            bot.send_message(chat_id, ('ساعت شروع' if key == 'night_start' else 'ساعت پایان') + ' ذخیره شد.', reply_markup=ADMIN_KB)
            st['awaiting'] = None
            return

    # Only capture when we are awaiting resume text in resume flow
    if st.get('flow') == 'resume' and st.get('awaiting') == 'resume_text' and st.get('county') and st.get('category'):
        resume_text = message.text.strip()
        # Basic sanity check to avoid capturing commands or empty text
        if not resume_text or resume_text.startswith('/'):
            bot.reply_to(message, 'متن رزومه معتبر نیست. لطفاً رزومه را به‌صورت متن معمولی ارسال کنید.')
            return

        db_save_resume(chat_id, st['county'], st['category'], resume_text)
        st['awaiting'] = None
        st['flow'] = None
        bot.send_message(
            chat_id,
            'رزومه شما با موفقیت ثبت شد. اعتبار رزومه یک هفته است. برای بازگشت به منو از دکمه‌ها استفاده کنید.',
            reply_markup=APPLICANT_KB
        )
        send_advertis(chat_id)
        return

    # Employer posting: capture job ad text
    if st.get('flow') == 'post_job' and st.get('awaiting') == 'job_text' and st.get('county') and st.get('category'):
        job_text = message.text.strip()
        if not job_text or job_text.startswith('/'):
            bot.reply_to(message, 'متن آگهی معتبر نیست. لطفاً متن آگهی را به‌صورت معمولی ارسال کنید.')
            return
        db_save_job_post(chat_id, st['county'], st['category'], job_text)
        st['awaiting'] = None
        st['flow'] = None
        bot.send_message(chat_id, 'آگهی شما ثبت شد و به مدت یک هفته فعال خواهد بود.', reply_markup=EMPLOYER_KB)
        send_advertis(chat_id)
        return

    # Otherwise, fall back guidance
    bot.reply_to(message, "برای شروع از /start استفاده کنید یا یکی از گزینه‌ها را از منوی کیبورد انتخاب نمایید.")


def send_resumes_for_county(chat_id: int, county: str):
    # housekeeping: purge expired records (> 7 days)
    try:
        db_cleanup_expired(7)
    except Exception:
        pass
    total = len(db_get_active_resumes_by_county(county))
    bot.send_message(chat_id, f'تعداد رزومه‌های فعال در "{county}": {total}')

    # Show 3 resumes only if more than 5 active
    if total > 5:
        remaining = db_get_remaining_views(chat_id)
        if remaining <= 0:
            bot.send_message(chat_id, 'سقف مشاهده روزانه شما (۳ رزومه) به اتمام رسیده است. لطفاً فردا مجدداً تلاش کنید.', reply_markup=EMPLOYER_KB)
            return
        unseen = db_get_unseen_resumes_by_county(chat_id, county)
        if not unseen:
            bot.send_message(chat_id, 'رزومه جدیدی برای نمایش وجود ندارد.', reply_markup=EMPLOYER_KB)
            return
        n = min(3, remaining)
        to_show = unseen[:n]
        for r in to_show:
            bot.send_message(chat_id, format_resume(r))
        db_add_views(chat_id, len(to_show))
        db_mark_resumes_seen(chat_id, [r['id'] for r in to_show])
        left = db_get_remaining_views(chat_id)
        bot.send_message(chat_id, f'تعداد رزومه‌های نمایش‌داده‌شده: {len(to_show)}. باقیمانده امروز: {left} از ۳.', reply_markup=EMPLOYER_KB)
    else:
        bot.send_message(chat_id, 'به‌دلیل کمتر بودن تعداد رزومه‌های فعال از ۶ مورد، نمایش رزومه انجام نشد.', reply_markup=EMPLOYER_KB)


if __name__ == '__main__':
    bot.infinity_polling()  # Blocks forever